/**
 * \file: feature_db.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: authorization level daemon
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2010, 2011 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include "create_signature_db_cmd/feature_db.h"
#include "create_signature_db_cmd/signature_db_writer.h"
#include "encryption/signature.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <limits.h>
#include <sys/stat.h>
#include <errno.h>

typedef struct feature_db_entry_t feature_db_entry_t;

typedef enum feature_search_result
{
	FEATURE_FOUND,
	FEATURE_NOT_FOUND,
	FEATURE_FOUND_TWICE
} feature_search_result;

typedef enum script_types
{
	ENABLE_PERMANENT=0,
	ENABLE_VOLATILE,
	DISABLE_PERMANENT,
	DISABLE_VOLATILE
}script_types;

typedef struct feature_db_entry_t
{
	bool found_marker;
	feature_db_entry_t *next_entry;
	char feature_name[];
} feature_db_entry_t;

static feature_db_entry_t *first_feature=NULL;

static error_code_t feature_db_add_feature(const char *feature_name);

static void feature_db_free_feature(feature_db_entry_t *feature);

static bool feature_db_check_feature_scripts(const char *feature_name, const char *root_dir, const char *sysroot_dir);

static error_code_t feature_db_get_script_path_and_validity(char *path, size_t max_path_len, const char *feature_name,
		const char *root_dir, const char *sysroot_dir, bool is_permanent, bool activated);



static void feature_db_start_completeness_check(void);
static feature_search_result feature_db_check_feature_present(const char *feature_name);
static bool feature_db_finalize_check(void);

static error_code_t feature_db_sign_feature_scripts(const char *feature_name, const char *root_dir);
static error_code_t feature_db_sign_feature_script(const char *path);
static error_code_t feature_db_sign_all_files_in_feature(char *path);

//--------------------------------------- functions for loading feature info and checking for script completeness -----

error_code_t feature_db_load(const char *root_dir, const char *sysroot_dir)
{
	error_code_t result=RESULT_OK;
	DIR *dp;
	struct dirent *ep;
	char feature_scripts_path[PATH_MAX];

	snprintf(feature_scripts_path,PATH_MAX,"%s%s%s/",sysroot_dir,root_dir,FEATURE_SUBDIR);
	dp = opendir (feature_scripts_path);
	if (dp == NULL)
	{
		printf("Unable to open the feature configuration in \'%s%s\'- %s.\n",sysroot_dir,root_dir,strerror(errno));
		return RESULT_INVALID;
	}

	while ((ep = readdir (dp))!=NULL && result==RESULT_OK)
	{
		if (strcmp(ep->d_name,"..")==0 || strcmp(ep->d_name,".")==0) continue;

		if (ep->d_type != DT_DIR)
		{
			printf("WARNING: Unexpected element found in feature conf directory: \'%s\'. Ignoring it.\n",ep->d_name);
			continue;
		}
		printf("Feature \'%s\' found. Scanning scripts of the feature ...\n",ep->d_name);
		if (!feature_db_check_feature_scripts(ep->d_name, root_dir, sysroot_dir))
		{
			printf("ERROR: No activation or deactivation scripts found at all."
					" What are you planning to do with this feature?\n");
			result=RESULT_INVALID;
			continue;
		}
		result=feature_db_add_feature(ep->d_name);
	}

	(void) closedir (dp);

	return result;
}

void feature_db_free(void)
{
	feature_db_entry_t *next_feature;
	while (first_feature!=NULL)
	{
		next_feature=first_feature->next_entry;
		feature_db_free_feature(first_feature);
		first_feature=next_feature;
	}
}

//--------------------------------------- functions for checking feature completeness in level files -----------------
error_code_t feature_db_check_level_for_feature_completeness(level_configuration_t *level_conf)
{
	const feature_t *feature;
	feature_db_start_completeness_check();

	feature=level_configuration_get_first_feature(level_conf);
	while (feature!=NULL)
	{
		const char *feature_name;
		feature_search_result search_result;

		feature_name=level_configuration_feature_name(feature);
		search_result=feature_db_check_feature_present(feature_name);
		if (search_result==FEATURE_FOUND_TWICE)
		{
			printf("ERROR: The security feature \'%s\' is listed more than ones in the level file of level \'%d\'.\n",
					feature_name,(int)level_configuration_get_security_level(level_conf));
			return RESULT_INVALID_LEVEL_CONFIGURATION;
		}
		else if (search_result==FEATURE_NOT_FOUND)
		{
			printf("ERROR: Unknown feature \'%s\' is listed in the level file of level \'%d\'.\n",
					feature_name,(int)level_configuration_get_security_level(level_conf));
			return RESULT_INVALID_LEVEL_CONFIGURATION;
		}
		feature=level_configuration_next_feature(feature);
	}

	if (!feature_db_finalize_check())
	{
		printf("ERROR: Not all security features are listed in the level file of level \'%d\'.\n",
				(int)level_configuration_get_security_level(level_conf));
		return RESULT_INVALID_LEVEL_CONFIGURATION;
	}

	return RESULT_OK;
}
//--------------------------------------- functions for filling the signature db with feature script signatures ------
error_code_t feature_db_fill_signature_db(const char *root_dir)
{
	error_code_t result=RESULT_OK;
	feature_db_entry_t *feature;

	feature=first_feature;
	while (feature!=NULL && result==RESULT_OK)
	{
		printf("Signing scripts of feature \'%s\' ...\n",feature->feature_name);
		result=feature_db_sign_feature_scripts(feature->feature_name,root_dir);
		feature=feature->next_entry;
	}

	return result;
}

//--------------------------------------- adding and freeing features ------------------------------------------------
static void feature_db_free_feature(feature_db_entry_t *feature)
{
	free(feature);
}

static error_code_t feature_db_add_feature(const char *feature_name)
{
	feature_db_entry_t *new_feature;
	//allocate memory for the fixed part of the DS plus the feature name plus '\0'
	new_feature=malloc(sizeof(feature_db_entry_t)+strlen(feature_name)+1);
	if (new_feature==NULL)
		return RESULT_NORESOURCE;
	new_feature->found_marker=false;
	strcpy(new_feature->feature_name, feature_name);

	//put it into the list
	new_feature->next_entry=first_feature;
	first_feature=new_feature;

	return RESULT_OK;
}

//--------------------------------------------------------------------------------------------------------------------

//----------------------------------------- script evaluation --------------------------------------------------------
static bool feature_db_check_feature_scripts(const char *feature_name, const char *root_dir, const char *sysroot_dir)
{
	char path[PATH_MAX];
	bool one_found=false;
	error_code_t result = RESULT_OK;

	// Here during scanning of feature scripts, it just checks for the availability of scripts
	// The file type check on scripts is not considered at this point.

	//activate permanent
	result = feature_db_get_script_path_and_validity(path,PATH_MAX,feature_name,root_dir,sysroot_dir,true,true);
	if ((result == RESULT_OK) || (result == RESULT_INVALID_FILE_TYPE))
		one_found=true;

	//activate volatile
	result = feature_db_get_script_path_and_validity(path,PATH_MAX,feature_name,root_dir,sysroot_dir,false,true);
	if ((result == RESULT_OK) || (result == RESULT_INVALID_FILE_TYPE))
		one_found=true;

	//deactivate permanent
	result = feature_db_get_script_path_and_validity(path,PATH_MAX,feature_name,root_dir,sysroot_dir,true,false);
	if ((result == RESULT_OK) || (result == RESULT_INVALID_FILE_TYPE))
		one_found=true;

	//deactivate volatile
	result = feature_db_get_script_path_and_validity(path,PATH_MAX,feature_name,root_dir,sysroot_dir,false,false);
	if ((result == RESULT_OK) || (result == RESULT_INVALID_FILE_TYPE))
		one_found=true;

	return one_found;
}
//--------------------------------------------------------------------------------------------------------------------

//--------------------------------------- feature script path generation ---------------------------------------------
//The output variable path could be a directory if exists or a script if there is no directory
static error_code_t feature_db_get_script_path_and_validity(char *path, size_t max_path_len, const char *feature_name,
		const char *root_dir, const char *sysroot_dir, bool is_permanent, bool activated)
{
	error_code_t result = RESULT_OK;
	const char *script_name;
	char file_path[PATH_MAX] = {0};
	struct stat stat_result;

	if (activated)
	{
		if (is_permanent)
			script_name=FEATURE_ENABLE_PERM_DIR;
		else
			script_name=FEATURE_ENABLE_VOLA_DIR;
	}
	else
	{
		if (is_permanent)
			script_name=FEATURE_DISABLE_PERM_DIR;
		else
			script_name=FEATURE_DISABLE_VOLA_DIR;
	}

	/*
	 * Non-directory cases - Inside feature directory but not inside feature sub-directory:
	 * -----------------------------------------------------------------------------------
	 * 	Case 1: If a script is missing,then it is intended. Signature db (sigdb) should be generated.
	 * 	Case 2: If a script is available but it is not of regular file type, then sigdb generation should fail.
	 * 	Case 3: If a regular file is having the same naming scheme as sub-directory,then signature db
	 * 			generation should fail.
	 *			e.g. In kernel_console feature, if a regular file "../security_features/kernel_console/enable_permanent"
	 *				exist. It should be having name as "enable_permanent.sh" since it is regular file instead it is having
	 *				name as "enable_permanent" like a directory name.
	 *
	 * Directory cases - Only inside feature sub-directory:
	 * ---------------------------------------------------
	 *	Case 4: If a sub-directory is available but its file type is not a directory,then sigdb generation should fail.
	 *	Case 5: If a directory type file is having the same naming scheme of scripts then sigdb generation should fail.
	 *			e.g. In automounter feature, "../security_features/automounter_restrictions/disable_permanent.sh/" is
	 *				sub-directory. The file name should be "disable_permanent", instead it is "disable_permanent.sh".
	 *	Case 6:	If a script having same name as directory is missing inside a directory, then sigdb generation should fail.
	 *			e.g. In automounter feature, "../security_features/automounter_restrictions/disable_permanent"
	 *				directory should contain disable_permanent.sh.
	 * 	Case 7:	Signature db generation mandates all the files inside feature sub-directory of regular file type,
	 * 			Failing this condition will fail sigdb generation.
	 *
	 * The different case numbers are commented in the below if conditions in the places where they are done.
	 *
	 */

	snprintf(path,max_path_len,"%s%s%s/%s/%s",sysroot_dir,root_dir,FEATURE_SUBDIR,feature_name,script_name);

	if (lstat(path,&stat_result) == -1)
	{
		if(errno == ENOENT)
		{
			snprintf(path,max_path_len,"%s%s%s/%s/%s%s",sysroot_dir,root_dir,FEATURE_SUBDIR,feature_name,script_name,SCRIPT_EXTN);
			if (lstat(path,&stat_result) == -1)
			{
				if (errno == ENOENT) //Case 1
				{
					result = RESULT_FILE_NOT_FOUND;
					printf("WARNING: \'%s\' script not found.\n",path);
				}
				else
				{
					result = RESULT_STAT_ERROR;
					 printf("ERROR: Unexpected stat error for path \'%s\' - %s.",path,strerror(errno));
				}
			}
			else
			{
				if (!S_ISREG(stat_result.st_mode)) //Case 2 & 5
				{
					result = RESULT_INVALID_FILE_TYPE;
					printf("ERROR: Unexpected file type for file \'%s\' - Expected a regular file type.\n",path);
				}
			}
		}
		else
		{
			result = RESULT_STAT_ERROR;
			printf("ERROR: Unexpected stat error for path \'%s\' - %s.",path,strerror(errno));
		}
	}
	else if (S_ISDIR(stat_result.st_mode))
	{
		snprintf(file_path,PATH_MAX,"%s/%s%s",path,script_name,SCRIPT_EXTN);
		if (lstat(file_path,&stat_result) == -1)
		{
			if (errno == ENOENT) //Case 6
			{
				result = RESULT_MUST_FILE_NOT_FOUND;
				printf("ERROR: \'%s\' script not found in the script folder.\n",file_path);
			}
			else
			{
				result = RESULT_STAT_ERROR;
				printf("ERROR: Unexpected stat error for path \'%s\' - %s.",file_path,strerror(errno));
			}
		}
		else
		{
			if (!S_ISREG(stat_result.st_mode)) //Case 7
			{
				result = RESULT_INVALID_FILE_TYPE;
				printf("ERROR: Unexpected file type for file \'%s\' - Expected a regular file type.\n",file_path);
			}
		}
	}
	else //Case 3 & 4
	{
		printf("ERROR: Unexpected file type for file \'%s\' - Expected a directory file type.\n",path);
		result = RESULT_INVALID_FILE_TYPE;
	}

	return result;
}

//--------------------------------------- level checking -------------------------------------------------------------
static void feature_db_start_completeness_check(void)
{
	feature_db_entry_t *entry;
	entry=first_feature;
	while (entry!=NULL)
	{
		entry->found_marker=false;
		entry=entry->next_entry;
	}
}

static feature_search_result feature_db_check_feature_present(const char *feature_name)
{
	feature_db_entry_t *entry;
	entry=first_feature;
	while (entry!=NULL)
	{
		if (strcmp(entry->feature_name,feature_name)==0)
		{
			//we found the feature
			if (entry->found_marker)
				return FEATURE_FOUND_TWICE;
			else
			{
				entry->found_marker=true;
				return FEATURE_FOUND;
			}
		}

		entry=entry->next_entry;
	}
	return FEATURE_NOT_FOUND;
}

static bool feature_db_finalize_check(void)
{
	bool result=true;

	//searching for entries that have not been marked 'true', which means that we found unlisted
	//feature during the checking
	feature_db_entry_t *entry;
	entry=first_feature;
	while (entry!=NULL)
	{
		if (!entry->found_marker)
		{
			result=false;
			printf("ERROR: Feature \'%s\' not listed in the level configuration file.\n",entry->feature_name);
		}
		entry=entry->next_entry;
	}

	return result;
}

static error_code_t feature_db_sign_all_files_in_feature(char *path)
{
	error_code_t result = RESULT_OK;
	struct dirent *ep;
	DIR *dp;
	char files_path[PATH_MAX];

	dp = opendir (path);
	if (dp == NULL)
	{
		printf("Unable to open the feature dir in \'%s\'- %s.\n",path,strerror(errno));
		return RESULT_INVALID;
	}

	while ((ep = readdir (dp))!=NULL && result==RESULT_OK)
	{
		if (strcmp(ep->d_name,"..")==0 || strcmp(ep->d_name,".")==0) continue;

		if (ep->d_type != DT_REG) //Case 7
		{
			printf("ERROR: Unexpected element found in feature conf directory: \'%s\'.\n",ep->d_name);
			result = RESULT_INVALID_FILE_TYPE;
			break;
		}
		snprintf(files_path,PATH_MAX,"%s/%s",path,ep->d_name);
		printf("Found file \'%s\' in \'%s\' path.\n",files_path,path);

		result=feature_db_sign_feature_script(files_path);
	}
	return result;
}

//--------------------------------------------------------------------------------------------------------------------

//----------------------------------- signature creation -------------------------------------------------------------
static error_code_t feature_db_sign_feature_scripts(const char *feature_name, const char *root_dir)
{
	error_code_t result=RESULT_OK;
	char path[PATH_MAX]={0};
	struct stat stat_result;
	script_types feat=ENABLE_PERMANENT;

	// Here during signing of feature scripts, the file availability and file type checks are applicable

	while((result == RESULT_OK) && (feat <= DISABLE_VOLATILE)  )
	{
		switch(feat)
		{
			case ENABLE_PERMANENT:
				result = feature_db_get_script_path_and_validity(path,PATH_MAX,feature_name,root_dir,"",true,true);
				break;
			case ENABLE_VOLATILE:
				result = feature_db_get_script_path_and_validity(path,PATH_MAX,feature_name,root_dir,"",false,true);
				break;
			case DISABLE_PERMANENT:
				result = feature_db_get_script_path_and_validity(path,PATH_MAX,feature_name,root_dir,"",true,false);
				break;
			case DISABLE_VOLATILE:
				result = feature_db_get_script_path_and_validity(path,PATH_MAX,feature_name,root_dir,"",false,false);
				break;
			default:
				/*Do Nothing*/
				break;
		}
		// At this point, if a script/sub-directory in a feature is missing - it is intended
		if (result == RESULT_OK )
		{
			if (stat(path,&stat_result) != -1)
			{
				if (S_ISDIR(stat_result.st_mode))
				{
					printf("Found directory \'%s\' in feature \'%s\'.\n", path, feature_name);
					result = feature_db_sign_all_files_in_feature(path);
				}
				else
				{
					printf("Found script \'%s\' in feature \'%s\'.\n", path, feature_name);
					result =  feature_db_sign_feature_script(path);
				}
			}
		}
		else
		{
			if (result == RESULT_FILE_NOT_FOUND)
				result = RESULT_OK;
		}

		feat++;
	}
	return result;
}

static error_code_t feature_db_sign_feature_script(const char *path)
{
	error_code_t result;

	result=signature_db_writer_add_from_file(path);
	//scripts might miss, we checked before that all files needed are present
	if (result==RESULT_FILE_NOT_FOUND)
		return RESULT_OK;
	if (result==RESULT_OK)
		printf("\tSignature created for \'%s\'\n",path);

	return result;
}
//--------------------------------------------------------------------------------------------------------------------
